/* ---------------------------------------------------------------------------
    2016 HID Global Corporation/ASSA ABLOY AB.  All rights reserved.

   Redistribution and use in source and binary forms, with or without modification,
   are permitted provided that the following conditions are met:
      - Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      - Redistributions in binary form must reproduce the above copyright notice,
        this list of conditions and the following disclaimer in the documentation
        and/or other materials provided with the distribution.
        THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
        AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
        THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
        ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
        FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
        (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
        LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
        ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
        (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
        THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 --------------------------------------------------------------------------- */
 
#include "stdafx.h"

#include "Scenarios.h"
#include "Scenarios.UI.h"
#include "Helper.h"


//-----------------------------------------------------------
//	 Function: Preamble
//
//	Initialize the PKCS environment and get the slot selected
//		return	CK_RV					 : Return code
//		param	[OU] CK_SLOT_ID* pSlotID : The slot number
//-----------------------------------------------------------
CK_RV Preamble(CK_SLOT_ID* pSlotID)
{
	/* Local variables */
	CK_RV rc = 0;
	CK_FUNCTION_LIST_PTR pFunctionList = NULL_PTR;
	TCHAR szUserEntry[BUFSIZ];
	CK_ULONG ulSlotCount = 0;
	CK_SLOT_ID aSlotID[MAX_SLOT_NUM];
	BOOL bNum = TRUE;
	unsigned long i;
	CK_SLOT_INFO SlotInfo;
	CK_SLOT_ID SlotID;
	
	/* Initalizes the Cryptoki library */
	rc = C_GetFunctionList(&pFunctionList);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetFunctionList failed with error ", rc);
		goto EndFunction;
	} 

	rc = C_Initialize(NULL_PTR);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Initialize failed with error ", rc);
		goto EndFunction;
	} 

	/* Obtain the count of slots in the system */

	/* the tokenPresent parameter is set to false so that we retrieve all 
	   the slots present on the system and not only those having a token in it */
	rc = C_GetSlotList(FALSE, NULL_PTR, &ulSlotCount);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetSlotList failed with error ", rc);
		ulSlotCount = 0;
	}
	
	if (rc != CKR_OK) 
	{
		fwprintf (stdout, _T("\nCould not retrieve Slot Count"));
		goto EndFunction;
	}
	else 
	{
		fwprintf (stdout, _T("\n\nFound %ld slot(s)\n"), ulSlotCount);
		if (ulSlotCount == 0)
		{
			rc = CKR_GENERAL_ERROR;
			fwprintf ( stdout, _T("\nInsert a slot before selecting a scenario\n"));
			goto EndFunction;
		}
	}
		
	/* Check for number of slots */
	if (ulSlotCount > MAX_SLOT_NUM)
	{
		fwprintf (stdout, _T("\nGeneral Failure : too many slots"));
		fwprintf (stdout, _T("\nFound %ld slot (max is %ld)"), ulSlotCount, MAX_SLOT_NUM);
		rc = CKR_GENERAL_ERROR;
		goto EndFunction;
	}

	/* Obtain a list of slots in the system */

	/* the tokenPresent parameter is set to false so that we retrieve all 
	   the slots present on the system and not only those having a token in it */
	rc = C_GetSlotList(FALSE, aSlotID, &ulSlotCount);
	if (rc != CKR_OK) 
	{
		fwprintf (stdout, _T("\n\tC_GetSlotList | list failed with error 0x%x"), rc);
	} 
	
	if (rc != CKR_OK) 
	{
		fwprintf (stdout, _T("\nCould not retrieve Slot List"));
		goto EndFunction;
	}

	
	/* Get the slot the user wants to work with */
	SlotID = (CK_SLOT_ID)-1;
	do 
	{
		fwprintf (stdout, __T("\nSelect a slot ID:\n"));
		for (i=0; i<ulSlotCount; i++)
		{
			/* Obtains information about a particular slot in the system */
			rc =  C_GetSlotInfo(aSlotID[i], &SlotInfo);
			if (rc != CKR_OK) 
			{
				DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetSlotInfo failed with error ", rc);
			}
			if (rc != CKR_OK) 
			{
				fwprintf (stdout, _T("\nCould not retrieve slot information"));
				continue;
			}
			if (rc != CKR_OK) 
			{
				rc= CKR_GENERAL_ERROR;
				goto EndFunction;
			} 

			/* Display information about a particular slot in the system */
			fwprintf(stdout, _T("\n\t- %lu. "), i+1);
			DisplayInfo((char*)SlotInfo.slotDescription, 64);
		}


		GetString (_T("\nEnter your choice"), szUserEntry, sizeof (szUserEntry));
		ToUpper(szUserEntry);
					
		/* numeric entry ? */
		bNum = TRUE;
		for (i=0; i < _tcsnbcnt((const wchar_t *)szUserEntry, _tcsclen((const wchar_t *)szUserEntry)+1) && bNum == TRUE; i++)
		{
			if (!isdigit(szUserEntry[i]))
				bNum = FALSE;
		}
		if (bNum == TRUE)
		{
			/* get menu choice */
			swscanf_s (szUserEntry, _T("%d"), &SlotID);
			SlotID = SlotID - 1;
		}
		
		for (i=0; i<ulSlotCount; i++)
		{
			if(aSlotID[i] == SlotID)
				break;
		}
		if(i == ulSlotCount) // not found
		{
			fwprintf (stdout, _T("\nEnter available Slot ID\n"));
			SlotID = (CK_SLOT_ID)-1;		
		}

	} while (SlotID == -1);

	if (pSlotID) *pSlotID = SlotID;

EndFunction:
	return rc;
}

//---------------------------------------------------------------------------
//	 Function: Postamble
//
//	Remove the PKCS environment
//		return	CK_RV							: Return code
//		param	[IN] CK_SESSION_HANDLE hSession : Session handle
//		param	[IN] BOOL bLogoutToDo			: True if Logout is necessary
//---------------------------------------------------------------------------
CK_RV Postamble(CK_SESSION_HANDLE hSession, BOOL bLogoutToDo)
{
	/* Local variables */
	CK_RV	rc = 0;

	if (hSession != 0)
	{
		if (bLogoutToDo)
		{
			rc = C_Logout(hSession);
			if (rc != CKR_OK) 
			{
				DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Logout failed with error ", rc);
			}
		}

		rc = C_CloseSession(hSession);
		if (rc != CKR_OK) 
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_CloseSession failed with error ", rc);
		}
	}

	rc = C_Finalize(NULL);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Finalize failed with error ", rc);
	} 

/* EndFunction: */
	fwprintf (stdout, _T("\n"));
	return rc;
}

//--------------------------------------------------------------------------------
//	 Function: OpenSession
//
//	Open the PKCS session
//		return	CK_RV					: Return code
//		param	[IN] CK_SLOT_ID SlotID	: Slot ID
//		param	[IN] BOOL bReadWrite	: True if Open Session in Read/Write mode
//--------------------------------------------------------------------------------
CK_RV OpenSession(CK_SLOT_ID SlotID, BOOL bReadWrite, CK_SESSION_HANDLE *phSession)
{
	/* Local variables */
	CK_RV	rc = 0;
	CK_FLAGS flags = 0;
	CK_SESSION_HANDLE hSession = 0;

	if (bReadWrite)
		flags = CKF_SERIAL_SESSION | CKF_RW_SESSION ;
	else
		flags = CKF_SERIAL_SESSION;

	rc = C_OpenSession(SlotID, flags, NULL_PTR, NULL_PTR, &hSession);
	if (rc != CKR_OK) 
	{
		if (bReadWrite)
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_OpenSession | RW failed with error ", rc);
		else
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_OpenSession | RO failed with error ", rc);
	} 

	if (phSession) *phSession = hSession;
	return rc;
}

//------------------------------------------------------------------------------------
//	 Function: ListCertificates
//
//	List the signing certificates
//		return	CK_RV							      : Return code
//		param	[IN] CK_SESSION_HANDLE hSession		  : Session
//		param	[OU] CK_OBJECT_HANDLE **pphObjectCert : List of certificate handles
//		param	[OU] int *pnbObjectCert				  : Number of certificate handles
//------------------------------------------------------------------------------------
CK_RV ListCertificates(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE **pphObjectCert, int *pnbObjectCert)
{
	/* Local variables */
	CK_OBJECT_HANDLE hObjectCert = 0;
	CK_ULONG ulObjectCount = 0;
	CK_OBJECT_CLASS	CertificateClass = CKO_CERTIFICATE;
	CK_BBOOL bToken = CK_TRUE;
	CK_ULONG ulCategory = 1;	// 1 = token user
	CK_ATTRIBUTE CertificateTemplate[] = 
	{
		{CKA_CLASS, NULL, sizeof(CertificateClass)},
		{CKA_TOKEN, NULL, sizeof(bToken)},
		{CKA_CERTIFICATE_CATEGORY, NULL, sizeof(ulCategory)},
	};
	int nCertificateTemplateElements = 3;
	CK_BYTE *CertLabel = NULL;
    CK_ATTRIBUTE CertLabelTemplate = {CKA_LABEL, NULL, 0};
	int nCert = 0;
	CK_OBJECT_HANDLE_PTR phObjectCert = NULL;
	CK_RV	rc = 0;

	CertificateTemplate[0].pValue = &CertificateClass;
	CertificateTemplate[1].pValue = &bToken;
	CertificateTemplate[2].pValue = &ulCategory;

	rc = C_FindObjectsInit(hSession, CertificateTemplate, nCertificateTemplateElements);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | CertificateTemplate failed with error ", rc);
		goto EndFunction;
	} 

	for(;;)
	{
		rc = C_FindObjects(hSession, &hObjectCert, 1, &ulObjectCount);
		if ((rc != CKR_OK) || (ulObjectCount == 0)) 
		{
			break;
		} 
		else 
		{
			nCert ++;
		}
	}

	if (nCert != 0)
	{
		phObjectCert = (CK_OBJECT_HANDLE_PTR)GlobalAlloc(GMEM_FIXED, nCert*sizeof(CK_OBJECT_HANDLE));
		if (phObjectCert) memset(phObjectCert, 0, nCert*sizeof(CK_OBJECT_HANDLE));
	}

	rc = C_FindObjectsFinal(hSession);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | CertificateTemplate failed with error ", rc);
		goto EndFunction;
	} 
	
	nCert = 0;

	rc = C_FindObjectsInit(hSession, CertificateTemplate, nCertificateTemplateElements);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | CertificateTemplate failed with error ", rc);
		goto EndFunction;
	} 
	
	for(;;)
	{
		rc = C_FindObjects(hSession, &hObjectCert, 1, &ulObjectCount);
		if ((rc != CKR_OK) || (ulObjectCount == 0)) 
		{
			if (rc != CKR_OK) DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | CertificateTemplate failed with error ", rc);
			break;
		} 
	
		/* Get Label of the certificate */
		CertLabelTemplate.pValue = NULL; 
		rc = C_GetAttributeValue(hSession, hObjectCert, &CertLabelTemplate, 1);

        if (rc == CKR_OK) 
        {
			CertLabelTemplate.pValue = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, (CertLabelTemplate.ulValueLen+1)*sizeof(CK_BYTE));
			if (CertLabelTemplate.pValue)
			{
				memset(CertLabelTemplate.pValue, 0, (CertLabelTemplate.ulValueLen+1)*sizeof(CK_BYTE));

				rc = C_GetAttributeValue(hSession, hObjectCert, &CertLabelTemplate, 1);

				if (rc == CKR_OK) 
				{			
					nCert++;

					/* Search for Signature keyword in the label */
					CertLabel = (CK_BYTE_PTR)CertLabelTemplate.pValue;

					fprintf(stdout,"\n\t- %lu. %s", nCert, CertLabel);

					phObjectCert[nCert-1] = hObjectCert;
				}
				else
					DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | CertLabelTemplate failed with error ", rc);
			  
				GlobalFree(CertLabelTemplate.pValue);
				CertLabelTemplate.pValue = NULL;
			}
		}
		else
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | CertLabelTemplate failed with error ", rc);
	}

	rc = C_FindObjectsFinal(hSession);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | CertificateTemplate failed with error ", rc);
		goto EndFunction;
	} 

	if (pphObjectCert)	*pphObjectCert = phObjectCert;

	if (pnbObjectCert) *pnbObjectCert = nCert;

EndFunction:
	return rc;
}

//------------------------------------------------------------------------
//	 Function: GetCertificateID
//
//	Get the Certificate ID 
//		return	CK_RV							  : Return code
//		param	[IN] CK_SESSION_HANDLE hSession	  : Session
//		param	[IN] CK_OBJECT_HANDLE hObjectCert : Certificate handle
//		param	[OU] CK_BYTE_PTR *ppCertId		  : Certificate ID
//		param	[OU] CK_ULONG *pCertIdLen		  : Certificate ID Length
//------------------------------------------------------------------------
CK_RV GetCertificateID(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObjectCert, CK_BYTE_PTR *ppCertId, CK_ULONG *pCertIdLen)
{
	/* Local variables */
	CK_ATTRIBUTE CertIdTemplate = {CKA_ID, NULL, 0};
	CK_RV	rc = 0;
		
	CertIdTemplate.pValue = NULL;
	rc = C_GetAttributeValue(hSession, hObjectCert, &CertIdTemplate, 1);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | CertIdTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	if (CertIdTemplate.ulValueLen == 0)
	{
		fwprintf (stdout, _T("\n\tC_GetAttributeValue CertIdTemplate.ulValueLen == 0"));
		goto EndFunction;
	}

	CertIdTemplate.pValue = (CK_BYTE_PTR)GlobalAlloc(GMEM_FIXED, CertIdTemplate.ulValueLen*sizeof(CK_BYTE));
	if (CertIdTemplate.pValue == NULL)
	{
		fwprintf (stdout, _T("\n\tGlobalAlloc returns CertIdTemplate.pValue == NULL"));
		goto EndFunction;
	}

	memset(CertIdTemplate.pValue,0,CertIdTemplate.ulValueLen);
	rc = C_GetAttributeValue(hSession, hObjectCert, &CertIdTemplate, 1);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue CertIdTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	if (ppCertId)	*ppCertId = (CK_BYTE_PTR)CertIdTemplate.pValue;
	if (pCertIdLen)	*pCertIdLen = CertIdTemplate.ulValueLen;

EndFunction:
	return rc;
}

//-----------------------------------------------------------
//	 Function: Login
//
//	Login to the card
//		return	CK_RV							: Return code
//		param	[IN] CK_SESSION_HANDLE hSession : Session
//		param	[IN] BOOL bContextSpecific		: Context
//-----------------------------------------------------------
CK_RV Login(CK_OBJECT_HANDLE hSession, BOOL bContextSpecific)
{
	/* Local variables */
	CK_RV	rc = 0;
	CK_UTF8CHAR_PTR pPIN = NULL;
	CK_ULONG ulPinLen = 0;
	CK_USER_TYPE ulUserType = CKU_USER;
	if (bContextSpecific) ulUserType = CKU_CONTEXT_SPECIFIC;

	GetPIN(_T("\nEnter your PIN (alphanumeric value)"), &pPIN, &ulPinLen);
		
	fwprintf (stdout, _T("\n\tLogon in progress ..."));
	rc = C_Login(hSession, ulUserType, pPIN, ulPinLen);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Login failed with error ", rc);
	}
	
	if (pPIN)
	{
		GlobalFree(pPIN);
		pPIN = NULL;
	}

	return rc;
}

//-------------------------------------------------------------------------
//	 Function: FindPrivateKey
//
//	Find private key
//		return	CK_RV								: Return code
//		param	[IN] CK_SESSION_HANDLE hSession		: Session
//		param	[IN] CK_BYTE_PTR pCertId			: Certificate ID
//		param	[IN] CK_ULONG ulCertIdLen			: Certificate ID length
//		param	[OU] CK_OBJECT_HANDLE *phPrivateKey : Private key
//-------------------------------------------------------------------------
CK_RV FindPrivateKey(CK_OBJECT_HANDLE hSession, CK_BYTE_PTR pCertId, CK_ULONG ulCertIdLen, CK_OBJECT_HANDLE *phPrivateKey)
{
	/* Local variables */
	CK_OBJECT_CLASS PrivateKeyClass = CKO_PRIVATE_KEY;
	CK_BBOOL bToken = CK_TRUE;
	CK_BBOOL bPrivate = CK_TRUE;
	CK_RV	rc = 0;
	CK_ULONG ulObjectCount = 0;
	CK_ATTRIBUTE PrivateKeyTemplate[] =
	{
		{CKA_TOKEN, NULL, sizeof(bToken)},
		{CKA_PRIVATE, NULL, sizeof(bPrivate)},
		{CKA_CLASS, NULL, sizeof(PrivateKeyClass)},
		{CKA_ID, NULL, 0},
	};
	CK_OBJECT_HANDLE hPrivateKey = 0;

		
	PrivateKeyTemplate[0].pValue = &bToken;
	PrivateKeyTemplate[1].pValue = &bPrivate;
	PrivateKeyTemplate[2].pValue = &PrivateKeyClass;
	PrivateKeyTemplate[3].pValue = pCertId;  
	PrivateKeyTemplate[3].ulValueLen = ulCertIdLen;

	rc = C_FindObjectsInit(hSession, PrivateKeyTemplate, 4);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | PrivateKeyTemplate failed with error ", rc);
		goto EndFunction;
	} 

	ulObjectCount = 0;

    rc = C_FindObjects(hSession, &hPrivateKey, 1, &ulObjectCount);
	if ((rc != CKR_OK) && (ulObjectCount != 0)) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | PrivateKeyTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	rc = C_FindObjectsFinal(hSession);
	if ((rc != CKR_OK) && (ulObjectCount != 0)) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | PrivateKeyTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	if (phPrivateKey) *phPrivateKey = hPrivateKey;

EndFunction:
	return rc;
}

//------------------------------------------------------------------------
//	 Function: SignData
//
//	Sign data
//		return	CK_RV								: Return code
//		param	[IN] CK_SESSION_HANDLE hSession		: Session
//		param	[IN] CK_OBJECT_HANDLE hPrivateKey	: Private key
//		param	[IN] CK_BYTE pData[]				: Data to sign
//		param	[IN] CK_ULONG ulDataLen				: Data length to sign
//		param	[OU] CK_BYTE **ppDataSigned			: Data signed
//		param	[OU] CK_ULONG *pulDataSignedlen		: Data signed length
//------------------------------------------------------------------------
CK_RV Signdata(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE hPrivateKey, CK_BYTE pData[], CK_ULONG ulDataLen, CK_BYTE **ppDataSigned, CK_ULONG *pulDataSignedlen)
{
	/* Local variables */
	CK_BBOOL bPINAlways = CK_TRUE;
	CK_ATTRIBUTE PinAlwaysTemplate = {CKA_ALWAYS_AUTHENTICATE, NULL, sizeof(bPINAlways)};
	CK_MECHANISM mechanism = {CKM_MD5_RSA_PKCS, NULL_PTR, 0};
	CK_BYTE *pDataSigned = NULL;
	CK_ULONG ulDataSignedlen = 0;

	int i = 0;
	CK_RV rc = 0;

	PinAlwaysTemplate.pValue = &bPINAlways;
	rc = C_GetAttributeValue(hSession, hPrivateKey, &PinAlwaysTemplate, 1);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue PinAlwaysTemplate failed with error ", rc);
		goto EndFunction;
	}
		
	/* Data Signing twice */
	for (i=0; i<2; i++)
	{
		rc = C_SignInit(hSession, &mechanism, hPrivateKey);
		if (rc != CKR_OK)
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_SignInit failed with error ", rc);
			goto EndFunction;
		}
		
		/* Check PIN Always ACR */
		if (bPINAlways == CK_TRUE) /*PIN ALWAYS ACR => PIN Prompted */
		{
			rc = Login(hSession, TRUE);
			if (rc != CKR_OK) goto EndFunction;
		}

		rc = C_Sign(hSession, pData, ulDataLen, NULL, &ulDataSignedlen);
		if (rc != CKR_OK)
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Sign failed with error ", rc);
			goto EndFunction;
		}
	
		/* Check PIN Always ACR */
		if (bPINAlways == CK_TRUE) /*PIN ALWAYS ACR => PIN Prompted */
		{
			rc = Login(hSession, TRUE);
			if (rc != CKR_OK) goto EndFunction;
		}

		if (pDataSigned)
		{
			GlobalFree(pDataSigned);
			pDataSigned = NULL;
		}
		pDataSigned = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, ulDataSignedlen*sizeof(CK_BYTE));
		memset(pDataSigned, 0, ulDataSignedlen);

		rc = C_Sign(hSession, pData, ulDataLen, pDataSigned, &ulDataSignedlen);
		if (rc != CKR_OK)
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Sign failed with error ", rc);
			goto EndFunction;
		}
		else
		{
			fwprintf (stdout, _T("\n\n\tData signed = "));
			DisplayByte((unsigned char*)pDataSigned, ulDataSignedlen);
		}
	}

	if (ppDataSigned) *ppDataSigned = pDataSigned;
	if (pulDataSignedlen) *pulDataSignedlen = ulDataSignedlen;

EndFunction:
	return rc;
}

//------------------------------------------------------------------------
//	 Function: FindPublicKey
//
//	Find public key
//		return	CK_RV								: Return code
//		param	[IN] CK_SESSION_HANDLE hSession		: Session
//		param	[IN] CK_BYTE_PTR	pCertId			: Certificate ID
//		param	[IN] CK_ULONG ulCertIdLen			: Certificate ID length
//		param	[OU] CK_OBJECT_HANDLE *phPublicKey	: Public key
//-------------------------------------------------------------------------
CK_RV FindPublicKey(CK_OBJECT_HANDLE hSession, CK_BYTE_PTR pCertId, CK_ULONG ulCertIdLen, CK_OBJECT_HANDLE *phPublicKey)
{
	/* Local variables */
	CK_BBOOL bToken = CK_TRUE;
	CK_BBOOL bVerify = CK_TRUE;
	CK_OBJECT_CLASS PublicKeyClass = CKO_PUBLIC_KEY;
	CK_ATTRIBUTE PublicKeyTemplate[] =
	{
		{CKA_TOKEN, NULL, sizeof(bToken)},
		{CKA_VERIFY, NULL, sizeof(bVerify)},
		{CKA_CLASS, NULL, sizeof(PublicKeyClass)},
		{CKA_ID, NULL, 0},
	};
	CK_OBJECT_HANDLE hPublicKey = 0;
	CK_ULONG ulObjectCount = 0;
	CK_RV rc = 0;

	PublicKeyTemplate[0].pValue = &bToken;
	PublicKeyTemplate[1].pValue = &bVerify;
	PublicKeyTemplate[2].pValue = &PublicKeyClass;
	PublicKeyTemplate[3].pValue = pCertId;  
	PublicKeyTemplate[3].ulValueLen = ulCertIdLen;

	rc = C_FindObjectsInit(hSession, PublicKeyTemplate, 4);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | PublicKeyTemplate failed with error ", rc);
		goto EndFunction;
	} 
	
	ulObjectCount = 0;

    rc = C_FindObjects(hSession, &hPublicKey, 1, &ulObjectCount);
	if ((rc != CKR_OK) && (ulObjectCount != 0)) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | PublicKeyTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	rc = C_FindObjectsFinal(hSession);
	if ((rc != CKR_OK) && (ulObjectCount != 0)) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | PublicKeyTemplate failed with error ", rc);
		goto EndFunction;
	}

	if (phPublicKey) *phPublicKey = hPublicKey;

EndFunction:
	return rc;
}

//----------------------------------------------------------------------
//	 Function:	VerifySignature
//
//	Verify signature
//		return	CK_RV							 : Return code
//		param	[IN] CK_SESSION_HANDLE hSession  : Session
//		param	[IN] CK_OBJECT_HANDLE hPublicKey : Public key
//		param	[IN] CK_BYTE pData[]			 : Data to verify
//		param	[IN] CK_ULONG ulDataLen			 : Data length to verify
//		param	[IN] CK_BYTE *pDataSigned		 : Data signed
//		param	[IN] CK_ULONG ulDataSignedlen	 : Data signed length
//---------------------------------------------------------------------
CK_RV VerifySignature(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE hPublicKey, CK_BYTE pData[], CK_ULONG ulDataLen, CK_BYTE *pDataSigned, CK_ULONG ulDataSignedlen)
{
	/* Local variables */
	CK_MECHANISM mechanism = {CKM_MD5_RSA_PKCS, NULL_PTR, 0};
	CK_RV rc = 0;

	rc = C_VerifyInit(hSession, &mechanism, hPublicKey);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_VerifyInit failed with error ", rc);
		goto EndFunction;
	}

	rc = C_Verify(hSession, pData, ulDataLen, pDataSigned, ulDataSignedlen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Verify failed with error ", rc);
		goto EndFunction;
	}

EndFunction:
	return rc;
}

//----------------------------------
//	 Function: DataSignatureScenario
//
//	Data signature scenario
//		return	void
//		param	void
//----------------------------------
void DataSignatureScenario(void)
{
	/* Local variables */
	CK_SLOT_ID SlotID = (CK_SLOT_ID)-1;
	CK_RV	rc = 0;
	CK_SESSION_HANDLE hSession = 0;
	CK_BYTE *CertId = NULL;
	CK_ULONG CertIdLen = 0;
	int nCert = 0;
	CK_OBJECT_HANDLE *phObjectCert = NULL;
	CK_OBJECT_HANDLE hObjectCert = 0;
	CK_OBJECT_HANDLE hPrivateKey = 0;
	CK_OBJECT_HANDLE hPublicKey = 0;
	TCHAR szUserEntry[BUFSIZ];
	BOOL bNum = TRUE;
	int nCertNumber = -1;
	CK_BYTE *pDataSigned = NULL;
	CK_ULONG ulDataSignedlen = 0;
	int i = 0;
	BOOL bLogoutToDo = FALSE;

	// Original message: Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.
	// MD5 algo is used for signature sample below, message hash is: 81 8c 6e 60 1a 24 f7 27 50 da 0f 6c 9b 8e be 28
	CK_BYTE pData[] = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.";
	CK_ULONG ulDataLen = sizeof(pData);

	fwprintf (stdout, _T("\n==================\n\nStarting Data signature scenario...\n"));
							
	/* Preamble */
	rc = Preamble(&SlotID);
	if (rc != CKR_OK) goto EndFunction;

	/* Open Session in read only mode */
	rc = OpenSession(SlotID, FALSE, &hSession);
	if (rc != CKR_OK) goto EndFunction;
	
	/* List the certificates present on the card and available for signing some data */
	fwprintf (stdout, _T("\n\nSelect a certificate:\n"));

	rc = ListCertificates(hSession, &phObjectCert, &nCert);
	if (rc != CKR_OK) goto EndFunction;

	if (nCert == 0)
	{
		fwprintf (stdout, _T("\n\tNo Signing Certificate found"));
		goto EndFunction;
	}

	GetString (_T("\nEnter your choice"), szUserEntry, sizeof (szUserEntry));
	ToUpper(szUserEntry);
					
	/* numeric entry ? */
	bNum = TRUE;
	for (i=0; i < (int)_tcsnbcnt((const wchar_t *)szUserEntry, _tcsclen((const wchar_t *)szUserEntry)+1) && bNum == TRUE; i++)
	{
		if (!isdigit(szUserEntry[i]))
			bNum = FALSE;
	}
	if (bNum == TRUE)
	{
		/* get menu choice */
		swscanf_s (szUserEntry, _T("%d"), &nCertNumber);
	}

	if ((nCertNumber > 0) && (nCertNumber <= nCert))
	{
		hObjectCert = phObjectCert[nCertNumber-1];

		/* Get the CKA_ID of the Certificate */
		rc = GetCertificateID(hSession, hObjectCert, &CertId, &CertIdLen);
		if (rc != CKR_OK) goto EndFunction;

		/* Promt for PIN */
		rc = Login(hSession, FALSE);
		if (rc != CKR_OK) goto EndFunction;
		else bLogoutToDo = TRUE;

		/* Identify the private key corresponding to the selected certificate */
		rc = FindPrivateKey(hSession, CertId, CertIdLen, &hPrivateKey);
		if (rc != CKR_OK) goto EndFunction;
		if (hPrivateKey == 0)
		{
			fwprintf (stdout, _T("\n\tPrivate Key not found"));
			goto EndFunction;
		}
		
		/* Sign data */
		rc = Signdata(hSession, hPrivateKey, pData, ulDataLen, &pDataSigned, &ulDataSignedlen);
		if (rc != CKR_OK) goto EndFunction;

		/* Identify the public key corresponding to the selected certificate */
		rc = FindPublicKey(hSession, CertId, CertIdLen, &hPublicKey);
		if (rc != CKR_OK) goto EndFunction;
		if (hPublicKey == 0)
		{
			fwprintf (stdout, _T("\n\tPublic Key not found"));
			goto EndFunction;
		}

		/* Verify the signature */
		rc = VerifySignature(hSession, hPublicKey, pData, ulDataLen, pDataSigned, ulDataSignedlen);
		if (rc != CKR_OK) goto EndFunction;
	 }
	 else
	 {
		 fwprintf (stdout, _T("\n\tWrong Certificate number"));
		 goto EndFunction;
	 }
		
EndFunction:
	if (CertId)
	{
		GlobalFree(CertId);
		CertId = NULL;
	}
	if (pDataSigned)
	{
		GlobalFree(pDataSigned);
		pDataSigned = NULL;
	}
	if (phObjectCert)
	{
		GlobalFree(phObjectCert);
		phObjectCert = NULL;
	}
	
	/* Postamble */
	rc = Postamble(hSession, bLogoutToDo);

	fwprintf (stdout, _T("\nData signature scenario completed.\n"));
}

//----------------------------------------------------------------------------------
//	 Function: CanCardBeUnblocked
//
//	Can Card be unblocked
//		return	CK_RV							 : Return code
//		param	[IN] CK_SESSION_HANDLE hSession  : Session
//		param	[OU] CK_OBJECT_HANDLE * phObject : Handle to unblock pin
//		param	[OU] CK_BBOOL * pbUnblocked		 : True if the card can be unblocked
//----------------------------------------------------------------------------------
CK_RV CanCardBeUnblocked(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE * phObject, CK_BBOOL *pbUnblocked)
{
	/* Local variables */
	CK_OBJECT_CLASS UnblockPin = CKO_UNBLOCK_PIN;
	CK_ATTRIBUTE UnblockPinTemplate = {CKA_CLASS, NULL, sizeof(UnblockPin)};
	CK_RV rc = 0;
	CK_ULONG ulObjectCount = 0;
	CK_OBJECT_HANDLE hObject = 0;

	UnblockPinTemplate.pValue = &UnblockPin;
	rc = C_FindObjectsInit(hSession, &UnblockPinTemplate, 1);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | UnblockPinTemplate failed with error ", rc);
		goto EndFunction;
	} 
	
	ulObjectCount = 0;

    rc = C_FindObjects(hSession, &hObject, 1, &ulObjectCount);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | UnblockPinTemplate failed with error ", rc);
		goto EndFunction;
	}
	
	rc = C_FindObjectsFinal(hSession);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | UnblockPinTemplate failed with error ", rc);
		goto EndFunction;
	}

	if (pbUnblocked)
	{
		if (ulObjectCount == 1) 
		{
			*phObject = hObject;
			*pbUnblocked = TRUE;
		}
		else *pbUnblocked = FALSE;
	}

EndFunction:
	return rc;
}

//--------------------------------------------------------------------
//	 Function: GetUnblockCode
//
//	Get the unblock code of the card
//		return	CK_RV							 : Return code
//		param	[IN] CK_SESSION_HANDLE hSession  : Session
//		param	[IN] CK_OBJECT_HANDLE hObject	 : Handle to unblock pin
//		param	[OU] CK_BYTE **ppusUnlock		 : Unlock code
//		param	[OU] CK_ULONG *pnUnlock			 : Unlock code length
//		param   [OU] CK_MECHANISM *pMechanism	 : Mechanism
//-------------------------------------------------------------------
CK_RV GetUnblockCode(CK_SESSION_HANDLE hSession, CK_OBJECT_HANDLE hObject, CK_BYTE **ppusUnlock, CK_ULONG *pnUnlock, CK_MECHANISM *pMechanism)
{
	/* Local variables */
	CK_ULONG ulChallengeRequirement = 0;
	CK_ATTRIBUTE ChallengeRequirementTemplate = {CKA_UNBLOCK_PIN_CHALLENGE_REQUIREMENT, NULL, sizeof(ulChallengeRequirement)};
	CK_MECHANISM Mechanism = { CKM_UNBLOCK_PIN_STATIC, NULL, 0};
	CK_BYTE usChallenge[255];
	CK_CHAR szUnlockCleaned[255];
	CK_ATTRIBUTE UnblockPinChallengeTemplate = {CKA_UNBLOCK_PIN_CHALLENGE, NULL, sizeof(usChallenge)};
	CK_BYTE *pusUnlock = NULL;
	CK_CHAR szUnlock[255];
	ULONG ulUnlockCodeLen = 0;
	CK_CHAR *pi = NULL;
	CK_CHAR *pj = NULL;
	int i = 0;
	CK_ULONG nUnlock = 0;
	TCHAR szUserEntry[BUFSIZ];
	CK_RV rc = 0;

	/* Static or Challenge/Response ? */
	ChallengeRequirementTemplate.pValue = &ulChallengeRequirement;
	rc = C_GetAttributeValue(hSession, hObject, &ChallengeRequirementTemplate, 1);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | ChallengeRequirementTemplate failed with error ", rc);
		goto EndFunction;
	}

	switch(ulChallengeRequirement)
	{
		case CK_OTP_PARAM_IGNORED:
			/* Static Unlock */
			fwprintf (stdout, _T("\n\tStatic Unlock"));
			Mechanism.mechanism = CKM_UNBLOCK_PIN_STATIC;
		
			/* Get Unlock code */
			GetString (_T("\nEnter Unlock code"), szUserEntry, (int)sizeof(szUserEntry));

			ulUnlockCodeLen = WideCharToMultiByte(CP_ACP, 0, szUserEntry, (int)_tcslen(szUserEntry), 0, 0, 0, 0);
			memset(szUnlock,0,sizeof(szUnlock));
			WideCharToMultiByte(CP_ACP, 0, szUserEntry, (int)_tcslen(szUserEntry), (LPSTR)szUnlock, ulUnlockCodeLen, 0, 0);
			break;

		case CK_OTP_PARAM_MANDATORY:
			/* Challenge/Response Unlock */
			fwprintf (stdout, _T("\n\tChallenge/Response Unlock.\n"));
			Mechanism.mechanism = CKM_UNBLOCK_PIN_DYNAMIC;
		
			UnblockPinChallengeTemplate.pValue = usChallenge;
			rc = C_GetAttributeValue(hSession, hObject, &UnblockPinChallengeTemplate, 1);
			if (rc != CKR_OK)
			{
				DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | UnblockPinChallengeTemplate failed with error ", rc);
				goto EndFunction;
			}
		
			fwprintf(stdout, _T("\n\tChallenge = "));
			DisplayByte((unsigned char*)usChallenge, UnblockPinChallengeTemplate.ulValueLen);
		
			/* get Unlock code */
			GetString (_T("\nEnter Response"), szUserEntry, (int)sizeof(szUserEntry));

			ulUnlockCodeLen = WideCharToMultiByte(CP_ACP, 0, szUserEntry, (int)_tcslen(szUserEntry), 0, 0, 0, 0);
			memset(szUnlock,0,sizeof(szUnlock));
			WideCharToMultiByte(CP_ACP, 0, szUserEntry, (int)_tcslen(szUserEntry), (LPSTR)szUnlock, ulUnlockCodeLen, 0, 0);
			break;

		default:
			fwprintf (stdout, _T("\n\tUnknown Challenge Requirement value."));
			goto EndFunction;
	}

	/* Check Unlock code - Remove the '-' if any */
	for ( pj=szUnlock, pi=szUnlockCleaned; pj < szUnlock + sizeof(szUnlock); pj++)
	{
		if (*pj != '-')
		{
			*pi++ = *pj;
		}
	}

	pusUnlock = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, 255);
	memset(pusUnlock, 0, 255);

	for (i = 0, nUnlock = 0; szUnlockCleaned[2*i] != '\0'; i++, nUnlock++)
	{
		if ( !IS_HEX(szUnlockCleaned[2*i]) ||
			 !IS_HEX(szUnlockCleaned[2*i+1]) ) 
		{
			fwprintf (stdout, _T("\n\tInvalid Unlock Code Format."));
			rc = CKR_ARGUMENTS_BAD;
			goto EndFunction;
		}
		pusUnlock[i] = (unsigned char)(CHAR_HEX (szUnlockCleaned[2*i + 1]) | (CHAR_HEX (szUnlockCleaned[2*i]) << 4));
	}

	if (ppusUnlock) *ppusUnlock = pusUnlock;
	if (pnUnlock) *pnUnlock = nUnlock;
	if (pMechanism) *pMechanism = Mechanism;

EndFunction:
	return rc;
}

//--------------------------------------------------------------------
//	 Function: UnblockCard
//
//	Unblock the card
//		return	CK_RV							 : Return code
//		param	[IN] CK_SESSION_HANDLE hSession  : Session
//		param	[IN] CK_MECHANISM Mechanism		 : Mechanism
//		param	[IN] CK_OBJECT_HANDLE hObject	 : Handle to unblock pin
//		param	[IN] CK_BYTE* pusUnlock			 : Unlock code
//		param	[IN] CK_ULONG nUnlock			 : Unlock code length
//-------------------------------------------------------------------
CK_RV UnblockCard(CK_SESSION_HANDLE hSession, CK_MECHANISM Mechanism, CK_OBJECT_HANDLE hObject, CK_BYTE* pusUnlock, CK_ULONG nUnlock)
{
	/* Local variables */
	CK_BYTE* pNewPIN = NULL;
	CK_ULONG ulNewPinLen = 0;
	CK_BYTE* pConfirmPIN = NULL;
	CK_ULONG ulConfirmPinLen = 0;
	CK_RV rc = 0;

	/* Get new pin code */
	GetPIN(_T("\nEnter new PIN (alphanumeric value)"), &pNewPIN, &ulNewPinLen);

	/* Confirm new pin code */
	GetPIN(_T("\nConfirm new PIN (alphanumeric value)"), &pConfirmPIN, &ulConfirmPinLen);

	if (ulNewPinLen != ulConfirmPinLen)
	{
		fwprintf (stdout, _T("\n\tPIN lengths are not the same"));
		goto EndFunction;
	}
	else if (memcmp(pNewPIN, pConfirmPIN, ulConfirmPinLen) != 0)
	{
		fwprintf (stdout, _T("\n\tPINs are not the same"));
		goto EndFunction;
	}

	/* Unblock the card */
	rc = C_VerifyUnblockPINInit(hSession, &Mechanism, hObject);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_VerifyUnblockPINInit failed with error ", rc);
		goto EndFunction;
	} 

	rc = C_VerifyUnblockPIN(hSession, pNewPIN, ulNewPinLen, pusUnlock, nUnlock);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_VerifyUnblockPIN failed with error ", rc);
		goto EndFunction;
	} 

	/* Verify the PIN */
	fwprintf (stdout, _T("\n\tLogon in progress ..."));
	rc = C_Login(hSession, CKU_USER, pNewPIN, ulNewPinLen);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Login failed with error ", rc);
		goto EndFunction;
	}
	
EndFunction:
	if (pNewPIN)
	{
		GlobalFree(pNewPIN);
		pNewPIN = NULL;
	}
	if (pConfirmPIN)
	{
		GlobalFree(pConfirmPIN);
		pNewPIN = NULL;
	}
	return rc;
}

//-------------------------------------------------------------
//	 Function: ChangePIN
//
//	Change the PIN
//		return	CK_RV							 : Return code
//		param	[IN] CK_SESSION_HANDLE hSession  : Session
//-------------------------------------------------------------
CK_RV ChangePIN(CK_SESSION_HANDLE hSession)
{
	/* Local variables */
	CK_BYTE* pPIN = NULL;
	CK_ULONG ulPinLen = 0;
	CK_BYTE* pNewPIN = NULL;
	CK_ULONG ulNewPinLen = 0;
	CK_BYTE* pConfirmPIN = NULL;
	CK_ULONG ulConfirmPinLen = 0;
	CK_RV rc = 0;

	/* Get pin code */
	GetPIN(_T("\nEnter your PIN (alphanumeric value)"), &pPIN, &ulPinLen);

	/* Get new pin code */
	GetPIN(_T("\nEnter new PIN (alphanumeric value)"), &pNewPIN, &ulNewPinLen);

	/* Confirm new pin code */
	GetPIN(_T("\nConfirm new PIN (alphanumeric value)"), &pConfirmPIN, &ulConfirmPinLen);

	if (ulNewPinLen != ulConfirmPinLen)
	{
		fwprintf (stdout, _T("\n\tPIN lengths are not the same"));
		goto EndFunction;
	}
	else if (memcmp(pNewPIN, pConfirmPIN, ulConfirmPinLen) != 0)
	{
		fwprintf (stdout, _T("\n\tPINs are not the same"));
		goto EndFunction;
	}

	rc = C_SetPIN(hSession, pPIN, ulPinLen, pNewPIN, ulNewPinLen);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_SetPIN failed with error ", rc);
		goto EndFunction;
	} 

	/* Verify the PIN */
	fwprintf (stdout, _T("\n\tLogon in progress..."));
	rc = C_Login(hSession, CKU_USER, pNewPIN, ulNewPinLen);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Login failed with error ", rc);
		goto EndFunction;
	}

EndFunction:
	if (pPIN)
	{
		GlobalFree(pPIN);
		pPIN = NULL;
	}
	if (pNewPIN)
	{
		GlobalFree(pNewPIN);
		pNewPIN = NULL;
	}
	if (pConfirmPIN)
	{
		GlobalFree(pConfirmPIN);
		pNewPIN = NULL;
	}
	return rc;
}

//--------------------------
//	 Function: LoginScenario
//
//	Login scenario
//		return	void
//		param	void
//--------------------------
void LoginScenario(void)
{
	/* Local variables */
	CK_SLOT_ID SlotID;
	CK_TOKEN_INFO TokenInfo;
	CK_SESSION_HANDLE hSession = 0;
	CK_SESSION_HANDLE hObject = 0;
	CK_BYTE *pusUnlock = NULL;
	CK_ULONG nUnlock = 0;
	CK_MECHANISM Mechanism = { CKM_UNBLOCK_PIN_STATIC, NULL, 0};
	CK_RV rc = 0;

	BOOL bLogoutToDo = FALSE;
	CK_BBOOL bUnblocked = FALSE;

	fwprintf (stdout, _T("\n==================\n\nStarting login scenario...\n"));

	/* Preamble */
	rc = Preamble(&SlotID);
	if (rc != CKR_OK) goto EndFunction;

	/* Open Session in read only mode */
	rc = OpenSession(SlotID, FALSE, &hSession);
	if (rc != CKR_OK) goto EndFunction;
	
	/* Obtains information from the token in the system */
	rc = C_GetTokenInfo(SlotID, &TokenInfo);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetTokenInfo failed with error ", rc);
		goto EndFunction;
	}

	/* Is the card locked ? */
	if (TokenInfo.flags & CKF_USER_PIN_LOCKED)	/* Card locked */
	{
		fwprintf (stdout, _T("\n\tCard locked "));

		/* The card can be unblocked ? */
		rc = CanCardBeUnblocked(hSession, &hObject, &bUnblocked);
		if (rc != CKR_OK) goto EndFunction;

		if (bUnblocked)
		{
			fwprintf (stdout, _T("\n\tThe card can be unblocked "));

			/* Get Unlocked code */
			rc = GetUnblockCode(hSession, hObject, &pusUnlock, &nUnlock, &Mechanism);
			if (rc != CKR_OK) goto EndFunction;

			rc = UnblockCard(hSession, Mechanism, hObject, pusUnlock, nUnlock);
			if (rc != CKR_OK) goto EndFunction;
			else bLogoutToDo = TRUE;
		}
		else
			fwprintf (stdout, _T("\n\tThe card cannot be unblocked "));
	}
	else
	{
		fwprintf (stdout, _T("\n\tCard not locked "));

		if (TokenInfo.flags & CKF_USER_PIN_TO_BE_CHANGED ) /* Change at first used flag */
		{
			fwprintf (stdout, _T("\n\tChange PIN at first used flag is set "));

			rc = ChangePIN(hSession);
			if (rc != CKR_OK) goto EndFunction;
			else bLogoutToDo = TRUE;
		}
		else
		{
			fwprintf (stdout, _T("\n\tChange PIN at first used flag is not set "));

			rc = Login(hSession, FALSE);
			if (rc != CKR_OK) goto EndFunction;
			else bLogoutToDo = TRUE;
		}
	}

EndFunction:
	if (pusUnlock)
	{
		GlobalFree(pusUnlock);
		pusUnlock = NULL;
	}

	/* Postamble */
	rc = Postamble(hSession, bLogoutToDo);

	fwprintf (stdout, _T("\nLogin scenario completed.\n"));
}

//--------------------------------------------------------------------
//	 Function: FindSecretKey
//
//	Find the secret key object
//		return	CK_RV							   : Return code
//		param	[IN] CK_SESSION_HANDLE hSession	   : Session
//		param	[IN] CK_UTF8CHAR *pLabel		   : Label
//		param   [IN] CK_ULONG ulLabelLen		   : Label length
//		param	[OU] CK_OBJECT_HANDLE *phSecretKey : Secret key object
//--------------------------------------------------------------------
CK_RV FindSecretKey(CK_OBJECT_HANDLE hSession, CK_UTF8CHAR *pLabel, CK_ULONG ulLabelLen, CK_OBJECT_HANDLE *phSecretKey)
{
	/* Local variables */
	CK_OBJECT_HANDLE hSecretKey = 0;
	CK_ULONG ulObjectCount = 0;
	CK_ATTRIBUTE SecretKeyTemplate = { CKA_LABEL, NULL, 0 };
	CK_RV rc = 0;

	SecretKeyTemplate.pValue = pLabel; 
	SecretKeyTemplate.ulValueLen = ulLabelLen; 
	rc = C_FindObjectsInit(hSession, &SecretKeyTemplate, 1);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | SecretKeyTemplate failed with error ", rc);
		goto EndFunction;
	}

	ulObjectCount = 0;

    rc = C_FindObjects(hSession, &hSecretKey, 1, &ulObjectCount);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | SecretKeyTemplate failed with error ", rc);
		goto EndFunction;
	}

	rc = C_FindObjectsFinal(hSession);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | SecretKeyTemplate failed with error ", rc);
		goto EndFunction;
	}

	if (ulObjectCount == 1)
	{
		if (phSecretKey) *phSecretKey = hSecretKey;
	}

EndFunction:
	return rc;
}

//--------------------------------------------------------------------------
//	 Function: Encrypt
//
//	Encrypt
//		return	CK_RV							   : Return code
//		param	[IN] CK_SESSION_HANDLE hSession	   : Session
//		param	[IN] CK_OBJECT_HANDLE hSecretKey   : Secret key object
//		param	[IN] CK_BYTE pData[]			   : Data to encrypt
//		param	[IN] CK_ULONG ulDataLen			   : Data length to encrypt
//		param	[OU] CK_BYTE **ppDataEncrypted	   : Data encrypted
//		param	[OU] CK_ULONG *pulDataEncryptedLen : Data encrypted length
//-------------------------------------------------------------------------
CK_RV Encrypt(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE hSecretKey, CK_BYTE pData[], CK_ULONG ulDataLen, CK_BYTE **ppDataEncrypted, CK_ULONG *pulDataEncryptedLen)
{
	/* Local variables */
	CK_BYTE IV[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
	CK_MECHANISM Mechanism = { CKM_DES3_CBC_PAD, NULL, sizeof(IV)};
	CK_ULONG ulDataEncryptedLen = 0;
	CK_BYTE *pDataEncrypted = NULL;
	CK_RV rc = 0;

	Mechanism.pParameter = IV;
	rc = C_EncryptInit(hSession, &Mechanism, hSecretKey);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_EncryptInit failed with error ", rc);
		goto EndFunction;
	}
	
	rc = C_Encrypt(hSession, pData, ulDataLen, NULL, &ulDataEncryptedLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Encrypt failed with error ", rc);
		goto EndFunction;
	}
	
	pDataEncrypted = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, ulDataEncryptedLen*sizeof(CK_BYTE));
	memset(pDataEncrypted, 0, ulDataEncryptedLen*sizeof(CK_BYTE));

	rc = C_Encrypt(hSession, pData, ulDataLen, pDataEncrypted, &ulDataEncryptedLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Encrypt failed with error ", rc);
		goto EndFunction;
	}

	if (ppDataEncrypted) *ppDataEncrypted = pDataEncrypted;
	if (pulDataEncryptedLen) *pulDataEncryptedLen = ulDataEncryptedLen;

EndFunction:
	return rc;
}

//--------------------------------------------------------------------------
//	 Function: Decrypt
//
//	Decrypt
//		return	CK_RV							   : Return code
//		param	[IN] CK_SESSION_HANDLE hSession	   : Session
//		param	[IN] CK_OBJECT_HANDLE hSecretKey   : Secret key object
//		param	[IN] CK_BYTE pDataEncrypted[]	   : Data to decrypt
//		param	[IN] CK_ULONG ulDataEncryptedLen   : Data length to decrypt
//		param	[OU] CK_BYTE **ppDataDecrypted	   : Data decrypted
//		param	[OU] CK_ULONG *pulDataDecryptedLen : Data decrypted length
//-------------------------------------------------------------------------
CK_RV Decrypt(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE hSecretKey, CK_BYTE pDataEncrypted[], CK_ULONG ulDataEncryptedLen, CK_BYTE **ppDataDecrypted, CK_ULONG *pulDataDecryptedLen)
{
	/* Local variables */
	CK_BYTE IV[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
	CK_MECHANISM Mechanism = { CKM_DES3_CBC_PAD, NULL, sizeof(IV)};
	CK_ULONG ulDataDecryptedLen = 0;
	CK_BYTE *pDataDecrypted = NULL;
	CK_RV rc = 0;

	Mechanism.pParameter = IV;
	rc = C_DecryptInit(hSession, &Mechanism, hSecretKey);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_DecryptInit failed with error ", rc);
		goto EndFunction;
	}

	rc = C_Decrypt(hSession, pDataEncrypted, ulDataEncryptedLen, NULL, &ulDataDecryptedLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Decrypt failed with error ", rc);
		goto EndFunction;
	}

	pDataDecrypted = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, ulDataDecryptedLen*sizeof(CK_BYTE));
	memset(pDataDecrypted, 0, ulDataDecryptedLen*sizeof(CK_BYTE));

	rc = C_Decrypt(hSession, pDataEncrypted, ulDataEncryptedLen, pDataDecrypted, &ulDataEncryptedLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Decrypt failed with error ", rc);
		goto EndFunction;
	}

	if (ppDataDecrypted) *ppDataDecrypted = pDataDecrypted;
	if (pulDataDecryptedLen) *pulDataDecryptedLen = ulDataDecryptedLen;

EndFunction:
	return rc;
}

//-----------------------------------
//	 Function: SecretKeyUsageScenario
//
//	Login Secret key usage scenario
//		return	void
//		param	void
//-----------------------------------
void SecretKeyUsageScenario(void)
{
	/* Local variables */
	CK_SLOT_ID SlotID;
	CK_SESSION_HANDLE hSession = 0;
	CK_RV rc = 0;
	CK_BYTE Key3DES[24] = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 
							0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
							0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28 };
	CK_ULONG ulKey3DES = 24;
	CK_OBJECT_CLASS	ObjectClass = CKO_SECRET_KEY;
	CK_BBOOL bToken = CK_TRUE;
	CK_BBOOL bPrivate = CK_TRUE;
	CK_UTF8CHAR ObjectLabel[] = "label";
	CK_KEY_TYPE	DesKeyType = CKK_DES3;
	CK_BYTE	DesFlag = {1};
	CK_ATTRIBUTE ObjectTemplate[] =
	{
		{CKA_TOKEN, NULL, sizeof(bToken)},
		{CKA_PRIVATE, NULL, sizeof(bPrivate)},
		{CKA_CLASS, NULL, sizeof(ObjectClass)},
		{CKA_LABEL, NULL, sizeof(ObjectLabel)},
		{CKA_KEY_TYPE, NULL, sizeof (DesKeyType)},
		{CKA_WRAP, NULL, sizeof (DesFlag)},
		{CKA_VALUE, NULL, sizeof(Key3DES)},
		{CKA_VALUE_LEN, NULL, sizeof(ulKey3DES)},
	};
	CK_OBJECT_HANDLE hObject = 0;
	CK_BYTE pData[] = {0x01, 0x02, 0x03, 0x04};
	CK_ULONG ulDataLen = sizeof(pData);
	CK_OBJECT_HANDLE hSecretKey = 0;
	
	CK_ULONG ulDataEncryptedLen = 0;
	CK_BYTE *pDataEncrypted = NULL;
	CK_ULONG ulDataDecryptedLen = 0;
	CK_BYTE *pDataDecrypted = NULL;
	BOOL bLogoutToDo = FALSE;

	fwprintf (stdout, _T("\n==================\n\nStarting secrete key usage scenario...\n"));
							
	/* Preamble */
	rc = Preamble(&SlotID);
	if (rc != CKR_OK)
		goto EndFunction;

	/* Open Session in read write mode */
	rc = OpenSession(SlotID, TRUE, &hSession);
	if (rc != CKR_OK) goto EndFunction;
	
	/* Promt for PIN */
	rc = Login(hSession, FALSE);
	if (rc != CKR_OK) goto EndFunction;
	else bLogoutToDo = TRUE;

	/* Create Object */
	fwprintf (stdout, _T("\n\n\tSecret Key creation ..."));
	ObjectTemplate[0].pValue = &bToken;
	ObjectTemplate[1].pValue = &bPrivate;
	ObjectTemplate[2].pValue = &ObjectClass;
	ObjectTemplate[3].pValue = ObjectLabel;
	ObjectTemplate[4].pValue = &DesKeyType;
	ObjectTemplate[5].pValue = &DesFlag;
	ObjectTemplate[6].pValue = Key3DES;
	ObjectTemplate[7].pValue = &ulKey3DES;
	rc = C_CreateObject(hSession, ObjectTemplate, 8, &hObject);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_CreateObject failed with error ", rc);
		goto EndFunction;
	} 
	
	/* Logout */
	rc = C_Logout(hSession);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Logout failed with error ", rc);
		goto EndFunction;
	}

	/* Signature with secret key */
	rc = Login(hSession, FALSE);
	if (rc != CKR_OK) goto EndFunction;
	else bLogoutToDo = TRUE;

	/* Search for the secret key object created before using its label */
	rc = FindSecretKey(hSession, ObjectLabel, sizeof(ObjectLabel), &hSecretKey);
	if (rc != CKR_OK) goto EndFunction;

	if (hSecretKey == 0)
	{
		fwprintf (stdout, _T("\n\tSecret Key not found"));
	}
	else
	{
		/* Encrypt the data with the secret key */
		rc = Encrypt(hSession, hSecretKey, pData, ulDataLen, &pDataEncrypted, &ulDataEncryptedLen);
		if (rc != CKR_OK) goto EndFunction;

		fwprintf (stdout, _T("\n\n\tData encrypted = "));
		DisplayByte((unsigned char*)pDataEncrypted, ulDataEncryptedLen);
		
		/* Decrypt the data with the secret key */
		rc = Decrypt(hSession, hSecretKey, pDataEncrypted, ulDataEncryptedLen, &pDataDecrypted, &ulDataDecryptedLen);

		fwprintf (stdout, _T("\n\n\tData decrypted = "));
		DisplayByte((unsigned char*)pDataDecrypted, ulDataDecryptedLen);

		/* Compare the decrypted data with the original data */
		if (ulDataDecryptedLen != ulDataLen)
		{
			fwprintf (stdout, _T("\n\tData lengths are not the same"));
			goto EndFunction;
		}
		else if (memcmp(pDataDecrypted, pData, ulDataDecryptedLen) != 0)
		{
			fwprintf (stdout, _T("\n\tDatas are not the same"));
			goto EndFunction;
		}

		/* Logout */
		rc = C_Logout(hSession);
		if (rc != CKR_OK) 
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Logout failed with error ", rc);
			goto EndFunction;
		}
	
		/* Secret Key object deletion */
		rc = Login(hSession, FALSE);
		if (rc != CKR_OK) goto EndFunction;
		else bLogoutToDo = TRUE;

		fwprintf (stdout, _T("\n\n\tSecret Key deletion ..."));
		rc = C_DestroyObject(hSession, hObject);
		if (rc != CKR_OK) 
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_DestroyObject failed with error ", rc);
			goto EndFunction;
		}
	}

EndFunction:
	if (pDataDecrypted)
	{
		GlobalFree(pDataDecrypted);
		pDataDecrypted = NULL;
	}
	if (pDataEncrypted)
	{
		GlobalFree(pDataEncrypted);
		pDataEncrypted = NULL;
	}

	/* Postamble */
	rc = Postamble(hSession, bLogoutToDo);

	fwprintf (stdout, _T("\nSecrete key usage scenario completed.\n"));
}

//-----------------------------------------------------------
//	 Function: FindOTPKey
//
//	Find the otp key
//		return	CK_RV							: Return code
//		param	[IN] CK_SESSION_HANDLE hSession	: Session
//		param	[OU] CK_OBJECT_HANDLE *phOTP	: OTP object
//-----------------------------------------------------------
CK_RV FindOTPKey(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE *phOTP)
{
	/* Local variables */
	CK_RV rc = 0;
	CK_OBJECT_CLASS	ObjectClass = CKO_OTP_KEY;
	CK_BBOOL bToken = CK_TRUE;
	CK_ATTRIBUTE OTPTemplate[] = 
	{ 
		{CKA_CLASS, NULL, sizeof(ObjectClass)},
		{CKA_TOKEN, NULL, sizeof(bToken)}
	};
	CK_OBJECT_HANDLE hOTP = 0;
	CK_ULONG ulObjectCount = 0;

	/* Find first OTP key on the token. */
	OTPTemplate[0].pValue = &ObjectClass;
	OTPTemplate[1].pValue = &bToken;
	rc = C_FindObjectsInit(hSession, OTPTemplate, 2);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsInit | OTPTemplate failed with error ", rc);
		goto EndFunction;
	} 
	
	ulObjectCount = 0;

    rc = C_FindObjects(hSession, &hOTP, 1, &ulObjectCount);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjects | OTPTemplate failed with error ", rc);
		goto EndFunction;
	}

	if (ulObjectCount == 0)
	{
		/* No OTP objects on the card */
		hOTP = 0;
		if (phOTP) *phOTP = hOTP;
		goto EndFunction;
	}

	rc = C_FindObjectsFinal(hSession);
	if (rc != CKR_OK) 
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_FindObjectsFinal | OTPTemplate failed with error ", rc);
		goto EndFunction;
	}

	if (phOTP) *phOTP = hOTP;

EndFunction:
	return rc;
}

//---------------------------------------------------------------
//	 Function: GetOTP
//
//	Get the otp object
//		return	CK_RV							: Return code
//		param	[IN] CK_SESSION_HANDLE hSession	: Session
//		param	[IN] CK_OBJECT_HANDLE  hOTPkey	: OTP Key
//		param	[OU] CK_BYTE **ppDataOTP		: OTP data
//		param	[OU] CK_ULONG *pulDataOTPLen	: OTP data length
//---------------------------------------------------------------
CK_RV GetOTP(CK_OBJECT_HANDLE hSession, CK_OBJECT_HANDLE hOTPKey, CK_BYTE **ppDataOTP, CK_ULONG *pulDataOTPLen)
{
	CK_ULONG ulDataOTPLen = 0;
	CK_BYTE *pDataOTP = NULL;
	CK_ULONG ulSignatureInfoLen = 0;
	CK_BYTE *pSignatureInfo = NULL;
	CK_MECHANISM Mechanism = { CKM_ACTI, NULL, 0 };
	CK_RV rc = 0;
	unsigned long i = 0;

	/* Sign to get the OTP value. */
	rc = C_SignInit(hSession, &Mechanism, hOTPKey);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_SignInit failed with error ", rc);
		goto EndFunction;
	}

	/* Get the buffer length needed for the OTP Value and any associated data. */
	rc = C_Sign(hSession, NULL, 0, NULL, &ulSignatureInfoLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Sign failed with error ", rc);
		goto EndFunction;
	}
	
	pSignatureInfo = (CK_BYTE_PTR) GlobalAlloc(GMEM_FIXED, ulSignatureInfoLen*sizeof(CK_BYTE));
	memset(pSignatureInfo, 0, ulSignatureInfoLen*sizeof(CK_BYTE));
	
	/* Get the actual OTP value and any associated data. */
	rc = C_Sign(hSession, NULL, 0, pSignatureInfo, &ulSignatureInfoLen);
	if (rc != CKR_OK)
	{
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Sign - OTP Generation failed with error ", rc);
		goto EndFunction;
	}

	/* Retrieve the OTP Value. After successful calls to C_Sign with an OTP mechanism, the pSignatureInfo parameter will be set to point to a CK_OTP_SIGNATURE_INFO structure. */
	for (i = 0; i < ((CK_OTP_SIGNATURE_INFO_PTR)pSignatureInfo)->ulCount; i++)
	{
		/* Looking for CK_OTP_VALUE flag */
		if (((CK_OTP_SIGNATURE_INFO_PTR)pSignatureInfo)->pParams[i].type == CK_OTP_VALUE)
		{
			ulDataOTPLen = ((CK_OTP_SIGNATURE_INFO_PTR)pSignatureInfo)->pParams[i].ulValueLen;

			pDataOTP = (CK_BYTE_PTR)GlobalAlloc(GMEM_FIXED, ulDataOTPLen*sizeof(CK_BYTE));
			memset(pDataOTP, 0, ulDataOTPLen*sizeof(CK_BYTE));

			memcpy(pDataOTP, (((CK_OTP_SIGNATURE_INFO_PTR)pSignatureInfo)->pParams[i].pValue), ulDataOTPLen*sizeof(CK_BYTE));
			break;
		}
	}
	
	if (pDataOTP == NULL)
	{
		rc = CKR_DEVICE_ERROR;
		DisplayErrorCode((CK_CHAR_PTR)"\n\tC_Sign - No OTP value found ", rc);
		goto EndFunction;
	}

	if (ppDataOTP) *ppDataOTP = pDataOTP;
	if (pulDataOTPLen) *pulDataOTPLen = ulDataOTPLen;

EndFunction:
	if (pSignatureInfo)
	{
		GlobalFree(pSignatureInfo);
		pSignatureInfo = NULL;
	}
	return rc;
}

//---------------------------------
//	 Function: OTPGenerationScenario
//
//	OTP generation scenario
//		return	void
//		param	void
//----------------------------------
void OTPGenerationScenario(void)
{
	/* Local variables */
	CK_SLOT_ID SlotID;
	CK_SESSION_HANDLE hSession = 0;
	CK_RV rc = 0;
	CK_OBJECT_HANDLE hOTPKey = 0;
	CK_ULONG ulChallengeRequirement = 0;
	CK_ATTRIBUTE ChallengeRequirementTemplate = {CKA_OTP_CHALLENGE_REQUIREMENT , NULL, sizeof(ulChallengeRequirement)};
	CK_ULONG ulDataOTPLen = 0;
	CK_BYTE *pDataOTP = NULL;
	BOOL bLogoutToDo = FALSE;

	fwprintf (stdout, _T("\n==================\n\nStarting OTP generation scenario...\n"));
							
	/* Preamble */
	rc = Preamble(&SlotID);
	if (rc != CKR_OK) goto EndFunction;

	/* Open Session in read mode */
	rc = OpenSession(SlotID, FALSE, &hSession);
	if (rc != CKR_OK) goto EndFunction;
	
	/* Promt for PIN */
	rc = Login(hSession, FALSE);
	if (rc != CKR_OK) goto EndFunction;
	else bLogoutToDo = TRUE;

	/* Search for objects of OTP type */
	rc = FindOTPKey(hSession, &hOTPKey);
	
	if (hOTPKey == 0)
	{
		fwprintf (stdout, _T("\n\tNo OTP objects on the card"));
	}
	else
	{
		/* Check only synchrone OTP supported */
		ChallengeRequirementTemplate.pValue = &ulChallengeRequirement;
		rc = C_GetAttributeValue(hSession, hOTPKey, &ChallengeRequirementTemplate, 1);
		if (rc != CKR_OK)
		{
			DisplayErrorCode((CK_CHAR_PTR)"\n\tC_GetAttributeValue | ChallengeRequirementTemplate failed with error ", rc);
			goto EndFunction;
		}
	
		if (ulChallengeRequirement == CK_OTP_PARAM_IGNORED)
		{
			fwprintf (stdout, _T("\n\tSynchronous OTP supported"));

			/* Get an OTP */
			rc = GetOTP(hSession, hOTPKey, &pDataOTP, &ulDataOTPLen);
			if (rc != CKR_OK) goto EndFunction;

			fwprintf (stdout, _T("\n\tOTP = "));
			DisplayInfo((char*)pDataOTP, ulDataOTPLen);
		}
		else
		{
			fwprintf (stdout, _T("\n\tAsynchronous OTP not supported"));
		}
	}

EndFunction:
	if (pDataOTP)
	{
		GlobalFree(pDataOTP);
		pDataOTP = NULL;
	}

	/* Postamble */
	rc = Postamble(hSession, bLogoutToDo);

	fwprintf (stdout, _T("\nOTP generation scenario completed.\n"));
}
